<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('gmc_device_approvals', function (Blueprint $table) {
            $table->id('approval_id');

            // User Reference
            $table->unsignedBigInteger('user_id');

            // Device Unique ID (Hash of hardware info)
            $table->string('device_unique_id', 100)->unique();

            // Device Information
            $table->string('device_name', 100)->nullable();
            $table->enum('device_type', ['DESKTOP', 'LAPTOP', 'MOBILE', 'TABLET', 'OTHER'])->default('DESKTOP');
            $table->enum('device_category', ['PRIMARY', 'SECONDARY', 'MOBILE', 'EMERGENCY'])->default('PRIMARY');

            // Hardware Information - Made nullable as requested
            $table->string('mac_address', 50)->nullable();
            $table->string('hdd_serial', 100)->nullable();
            $table->string('motherboard_serial', 100)->nullable();
            $table->string('processor_id', 100)->nullable();
            $table->string('browser_fingerprint', 255)->nullable();

            // Network & Software
            $table->string('ip_address', 45)->nullable();
            $table->string('operating_system', 100)->nullable();
            $table->string('browser', 100)->nullable();

            // Department - location removed as requested
            $table->unsignedBigInteger('department_id')->nullable();

            // Approval Details
            $table->timestamp('requested_at')->nullable();
            $table->unsignedBigInteger('requested_by')->nullable();
            $table->timestamp('approved_at')->nullable();
            $table->unsignedBigInteger('approved_by')->nullable();
            $table->enum('approval_type', ['MANUAL', 'AUTO_CITIZEN', 'EMERGENCY'])->default('MANUAL');

            // Validity (1 Year for Non-Citizen, Permanent for Citizen)
            // Changed to nullable and added default values
            $table->timestamp('valid_from')->nullable();
            $table->timestamp('valid_until')->nullable();

            // Status
            $table->enum('status', ['ACTIVE', 'INACTIVE', 'SUSPENDED', 'EXPIRED', 'REVOKED'])->default('ACTIVE');
            $table->boolean('is_active')->default(true);
            $table->boolean('is_primary_device')->default(true);

            // Timestamps
            $table->timestamps();

            // Foreign Keys
            $table->foreign('user_id')->references('user_id')->on('gmc_users')->onDelete('cascade');
            $table->foreign('requested_by')->references('user_id')->on('gmc_users')->onDelete('set null');
            $table->foreign('approved_by')->references('user_id')->on('gmc_users')->onDelete('set null');
            $table->foreign('department_id')->references('department_id')->on('departments')->onDelete('set null');

            // Indexes
            $table->index(['user_id', 'status']);
            $table->index('device_unique_id');
            $table->index('valid_until');
            $table->index('mac_address');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('gmc_device_approvals');
    }
};
