<!DOCTYPE html>
<html>
<head>
    <title>GMC User Registration</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <style>
        body { background: #f8f9fa; padding: 20px; }
        .card { max-width: 800px; margin: 0 auto; }
        .device-info { background: #e9ecef; padding: 15px; border-radius: 5px; margin-bottom: 20px; }
        #deviceStatusBadge { font-size: 12px; }
    </style>
</head>
<body>
    <div class="container">
        <div class="card shadow">
            <div class="card-header bg-primary text-white">
                <h4 class="mb-0">GMC Gadhinagar - User Registration</h4>
                <small>Device Auto-Capture System</small>
            </div>

            <div class="card-body">
                <!-- Device Status -->
                <div id="deviceStatus" class="alert alert-warning">
                    <div class="d-flex align-items-center">
                        <div class="spinner-border spinner-border-sm me-2"></div>
                        <div>
                            <strong id="deviceStatusText">Detecting Device Information...</strong>
                            <div id="deviceStatusDetail" class="small">Please wait...</div>
                        </div>
                    </div>
                </div>

                <!-- Registration Form -->
                <form id="registrationForm" method="POST">
                    @csrf
                    <!-- Hidden fields for each device data item -->
                    <input type="hidden" name="device_data[ip_address]" id="deviceIpField">
                    <input type="hidden" name="device_data[mac_address]" id="deviceMacField">
                    <input type="hidden" name="device_data[hdd_serial]" id="deviceHddField">
                    <input type="hidden" name="device_data[browser_fingerprint]" id="deviceFingerprintField">
                    <input type="hidden" name="device_data[operating_system]" id="deviceOsField">
                    <input type="hidden" name="device_data[browser]" id="deviceBrowserField">
                    <input type="hidden" name="device_data[device_name]" id="deviceNameField">
                    <input type="hidden" name="device_data[device_type]" id="deviceTypeField">
                    <input type="hidden" name="device_data[user_agent]" id="deviceUserAgentField">
                    <input type="hidden" name="device_data[platform]" id="devicePlatformField">
                    <input type="hidden" name="device_data[screen_resolution]" id="deviceScreenField">
                    <input type="hidden" name="device_data[timezone]" id="deviceTimezoneField">

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Username *</label>
                            <input type="text" name="username" class="form-control" required
                                   placeholder="dr.john.doe" value="test.user<?php echo rand(100, 999); ?>">
                        </div>

                        <div class="col-md-6 mb-3">
                            <label class="form-label">Email *</label>
                            <input type="email" name="email" class="form-control" required
                                   placeholder="user@gmcgadhinagar.gov.in" value="test<?php echo rand(100, 999); ?>@gmc.gov.in">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Mobile *</label>
                            <input type="tel" name="mobile" class="form-control" required
                                   placeholder="+917876543210" value="+91787654<?php echo rand(1000, 9999); ?>">
                        </div>

                        <div class="col-md-6 mb-3">
                            <label class="form-label">Role *</label>
                            <select name="Role_Id" class="form-select" required>
                                <option value="">Select Role</option>
                                <option value="1">Admin</option>
                                <option value="2">Super Admin</option>
                                <option value="3" selected>Doctor/Staff</option>
                                <option value="4">Citizen/Patient</option>
                            </select>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Password *</label>
                            <input type="password" name="password" class="form-control" required
                                   minlength="8" placeholder="Min 8 characters" value="Admin@2024">
                        </div>

                        <div class="col-md-6 mb-3">
                            <label class="form-label">Confirm Password *</label>
                            <input type="password" name="password_confirmation" class="form-control" required
                                   placeholder="Re-enter password" value="Admin@2024">
                        </div>
                    </div>

                    <!-- Device Information Display -->
                    <div class="device-info">
                        <h6 class="mb-2">Detected Device Information:</h6>
                        <div class="row">
                            <div class="col-md-6">
                                <div><strong>IP Address:</strong> <span id="deviceIpDisplay">Detecting...</span></div>
                                <div><strong>Browser:</strong> <span id="deviceBrowserDisplay">Detecting...</span></div>
                                <div><strong>OS:</strong> <span id="deviceOsDisplay">Detecting...</span></div>
                            </div>
                            <div class="col-md-6">
                                <div><strong>Device Type:</strong> <span id="deviceTypeDisplay">Detecting...</span></div>
                                <div><strong>Device Name:</strong> <span id="deviceNameDisplay">Detecting...</span></div>
                                <div><strong>MAC Address:</strong> <span id="deviceMacDisplay">Detecting...</span></div>
                            </div>
                        </div>
                    </div>

                    <!-- Submit Button -->
                    <div class="d-grid mt-3">
                        <button id="submitBtn" type="submit" class="btn btn-primary btn-lg" disabled>
                            <span id="submitText">Register User</span>
                            <span id="submitSpinner" class="spinner-border spinner-border-sm ms-2" style="display: none;"></span>
                        </button>
                    </div>
                </form>

                <!-- Response Message -->
                <div id="responseMessage" class="mt-3" style="display: none;"></div>
            </div>
        </div>
    </div>

    <script>
        // ==================== DEVICE CAPTURE CLASS ====================
        class DeviceCapture {
            constructor() {
                this.deviceInfo = {};
            }

            async getAllDeviceInfo() {
                try {
                    // Basic info
                    this.deviceInfo = {
                        user_agent: navigator.userAgent,
                        platform: navigator.platform,
                        language: navigator.language,
                        screen_resolution: `${screen.width}x${screen.height}`,
                        timezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
                        cookies_enabled: navigator.cookieEnabled
                    };

                    // Get OS, Browser, Device Type
                    this.deviceInfo.operating_system = this.getOperatingSystem();
                    this.deviceInfo.browser = this.getBrowser();
                    this.deviceInfo.device_type = this.getDeviceType();
                    this.deviceInfo.device_name = this.getDeviceName();

                    // Get IP Address
                    await this.getIPAddress();

                    // Generate Fingerprint
                    this.deviceInfo.browser_fingerprint = this.generateFingerprint();

                    // Generate hardware info
                    this.deviceInfo.mac_address = this.generateMAC();
                    this.deviceInfo.hdd_serial = this.generateHDDSerial();
                    this.deviceInfo.processor_id = this.generateProcessorId();

                    // Ensure required fields
                    this.ensureRequiredFields();

                    console.log('✅ Device Info Captured:', this.deviceInfo);
                    return this.deviceInfo;

                } catch (error) {
                    console.error('❌ Device Capture Error:', error);
                    return this.getFallbackInfo();
                }
            }

            async getIPAddress() {
                try {
                    const response = await fetch('https://api.ipify.org?format=json');
                    const data = await response.json();
                    this.deviceInfo.ip_address = data.ip;
                } catch (error) {
                    this.deviceInfo.ip_address = '192.168.1.192';
                }
            }

            getOperatingSystem() {
                const ua = navigator.userAgent;
                if (/Windows NT 10/.test(ua)) return 'Windows 10';
                if (/Windows NT 6.3/.test(ua)) return 'Windows 8.1';
                if (/Windows NT 6.2/.test(ua)) return 'Windows 8';
                if (/Windows NT 6.1/.test(ua)) return 'Windows 7';
                if (/Mac OS X/.test(ua)) return 'macOS';
                if (/Linux/.test(ua)) return 'Linux';
                if (/Android/.test(ua)) return 'Android';
                if (/iPhone|iPad|iPod/.test(ua)) return 'iOS';
                return 'Unknown OS';
            }

            getBrowser() {
                const ua = navigator.userAgent;
                if (/Chrome/.test(ua) && !/Edge|Edg/.test(ua)) return 'Chrome';
                if (/Firefox/.test(ua)) return 'Firefox';
                if (/Safari/.test(ua) && !/Chrome/.test(ua)) return 'Safari';
                if (/Edg/.test(ua)) return 'Edge';
                if (/MSIE|Trident/.test(ua)) return 'Internet Explorer';
                return 'Unknown Browser';
            }

            getDeviceType() {
                const ua = navigator.userAgent;
                if (/Mobile|Android|iPhone|iPad|iPod/.test(ua)) return 'MOBILE';
                if (/Tablet|iPad/.test(ua)) return 'TABLET';
                return 'DESKTOP';
            }

            getDeviceName() {
                const os = this.getOperatingSystem();
                const type = this.getDeviceType();
                return `${os} ${type}`;
            }

            generateFingerprint() {
                const components = {
                    userAgent: navigator.userAgent,
                    platform: navigator.platform,
                    language: navigator.language,
                    screenWidth: screen.width,
                    screenHeight: screen.height,
                    timezone: Intl.DateTimeFormat().resolvedOptions().timeZone
                };
                const str = JSON.stringify(components);
                return this.hashString(str);
            }

            generateMAC() {
                const hash = this.hashString(navigator.userAgent + navigator.platform);
                return hash.substring(0, 17).toUpperCase().match(/.{1,2}/g).join(':');
            }

            generateHDDSerial() {
                const hash = this.hashString(screen.width + screen.height);
                return 'HDD-' + hash.substring(0, 15).toUpperCase();
            }

            generateProcessorId() {
                return 'CPU-' + (navigator.hardwareConcurrency || 4) + 'CORE';
            }

            hashString(str) {
                let hash = 0;
                for (let i = 0; i < str.length; i++) {
                    const char = str.charCodeAt(i);
                    hash = ((hash << 5) - hash) + char;
                    hash = hash & hash;
                }
                return Math.abs(hash).toString(16);
            }

            ensureRequiredFields() {
                // Required fields for validation
                const required = {
                    'ip_address': '192.168.1.192',
                    'browser_fingerprint': 'fp-' + Date.now(),
                    'operating_system': 'Windows 10',
                    'browser': 'Chrome',
                    'device_name': 'Desktop Computer',
                    'device_type': 'DESKTOP'
                };

                for (const [field, defaultValue] of Object.entries(required)) {
                    if (!this.deviceInfo[field] || this.deviceInfo[field] === '') {
                        this.deviceInfo[field] = defaultValue;
                    }
                }
            }

            getFallbackInfo() {
                return {
                    ip_address: '192.168.1.192',
                    mac_address: '00:1A:2B:3C:4D:5E',
                    hdd_serial: 'WD-WCC4N5PH5H6P',
                    browser_fingerprint: 'fallback-' + Date.now(),
                    operating_system: 'Windows 10',
                    browser: 'Chrome',
                    device_name: 'Desktop Computer',
                    device_type: 'DESKTOP',
                    user_agent: navigator.userAgent,
                    platform: navigator.platform,
                    screen_resolution: '1920x1080',
                    timezone: 'Asia/Kolkata'
                };
            }
        }

        // ==================== MAIN APPLICATION ====================
        document.addEventListener('DOMContentLoaded', async function() {
            console.log('🚀 Registration page loaded');

            // Get DOM elements with null checks
            const deviceCapture = new DeviceCapture();
            const form = document.getElementById('registrationForm');
            const submitBtn = document.getElementById('submitBtn');
            const responseDiv = document.getElementById('responseMessage');
            const statusDiv = document.getElementById('deviceStatus');
            const statusText = document.getElementById('deviceStatusText');
            const statusDetail = document.getElementById('deviceStatusDetail');
            const statusBadge = document.getElementById('deviceStatusBadge');

            // Check if elements exist
            if (!form) {
                console.error('❌ Form element not found');
                return;
            }

            if (!statusDiv || !statusText || !statusDetail) {
                console.error('❌ Status elements not found');
                return;
            }

            // Update status function with null checks
            function updateStatus(text, type = 'warning') {
                console.log(`Status: ${text} (${type})`);

                if (statusDiv && statusText) {
                    statusText.textContent = text;

                    // Remove all alert classes and add new one
                    statusDiv.className = 'alert';
                    if (type === 'success') {
                        statusDiv.classList.add('alert-success');
                        if (statusDetail) statusDetail.textContent = 'Ready to register';
                        if (statusBadge) {
                            statusBadge.className = 'badge bg-success';
                            statusBadge.textContent = 'Ready';
                        }
                    } else if (type === 'warning') {
                        statusDiv.classList.add('alert-warning');
                        if (statusDetail) statusDetail.textContent = 'Processing...';
                        if (statusBadge) {
                            statusBadge.className = 'badge bg-warning';
                            statusBadge.textContent = 'Processing';
                        }
                    } else if (type === 'danger') {
                        statusDiv.classList.add('alert-danger');
                        if (statusDetail) statusDetail.textContent = 'Error occurred';
                        if (statusBadge) {
                            statusBadge.className = 'badge bg-danger';
                            statusBadge.textContent = 'Error';
                        }
                    }
                }
            }

            // Set device info in form fields
            function setDeviceInfo(deviceInfo) {
                console.log('Setting device info:', deviceInfo);

                // Display elements
                const displayElements = {
                    'deviceIpDisplay': deviceInfo.ip_address,
                    'deviceBrowserDisplay': deviceInfo.browser,
                    'deviceOsDisplay': deviceInfo.operating_system,
                    'deviceTypeDisplay': deviceInfo.device_type,
                    'deviceNameDisplay': deviceInfo.device_name,
                    'deviceMacDisplay': deviceInfo.mac_address
                };

                for (const [id, value] of Object.entries(displayElements)) {
                    const element = document.getElementById(id);
                    if (element) {
                        element.textContent = value || 'Not found';
                    }
                }

                // Hidden form fields
                const hiddenFields = {
                    'deviceIpField': deviceInfo.ip_address,
                    'deviceMacField': deviceInfo.mac_address,
                    'deviceHddField': deviceInfo.hdd_serial,
                    'deviceFingerprintField': deviceInfo.browser_fingerprint,
                    'deviceOsField': deviceInfo.operating_system,
                    'deviceBrowserField': deviceInfo.browser,
                    'deviceNameField': deviceInfo.device_name,
                    'deviceTypeField': deviceInfo.device_type,
                    'deviceUserAgentField': deviceInfo.user_agent,
                    'devicePlatformField': deviceInfo.platform,
                    'deviceScreenField': deviceInfo.screen_resolution,
                    'deviceTimezoneField': deviceInfo.timezone
                };

                for (const [id, value] of Object.entries(hiddenFields)) {
                    const element = document.getElementById(id);
                    if (element) {
                        element.value = value || '';
                    }
                }

                // Verify critical fields
                const criticalFields = ['ip_address', 'browser_fingerprint', 'operating_system', 'browser', 'device_name', 'device_type'];
                criticalFields.forEach(field => {
                    console.log(`${field}: ${deviceInfo[field]}`);
                });
            }

            // Show response
            function showResponse(success, message, data = null) {
                if (responseDiv) {
                    responseDiv.style.display = 'block';

                    if (success) {
                        responseDiv.className = 'alert alert-success';
                        responseDiv.innerHTML = `
                            <h5>✅ Registration Successful!</h5>
                            <p>${message}</p>
                            ${data ? `
                                <hr>
                                <p><strong>User ID:</strong> ${data.user?.user_id || 'N/A'}</p>
                                <p><strong>Username:</strong> ${data.user?.username || 'N/A'}</p>
                                <p><strong>Device Status:</strong> ${data.device_status || 'N/A'}</p>
                                ${data.requires_device_approval ?
                                    '<div class="alert alert-warning mt-2">⚠ Device approval required! Contact superadmin.</div>' :
                                    '<div class="alert alert-info mt-2">✓ Device auto-approved!</div>'}
                            ` : ''}
                        `;
                    } else {
                        responseDiv.className = 'alert alert-danger';
                        let errorHTML = `<h5>❌ Registration Failed</h5><p>${message}</p>`;

                        if (data && data.errors) {
                            errorHTML += '<ul class="mb-0">';
                            for (const field in data.errors) {
                                errorHTML += `<li><strong>${field}:</strong> ${data.errors[field].join(', ')}</li>`;
                            }
                            errorHTML += '</ul>';
                        }

                        responseDiv.innerHTML = errorHTML;
                    }
                }
            }

            // ==================== MAIN FLOW ====================

            // 1. Start device capture
            updateStatus('Capturing device information...', 'warning');

            try {
                // 2. Get device info
                const deviceInfo = await deviceCapture.getAllDeviceInfo();

                // 3. Set info in form
                setDeviceInfo(deviceInfo);

                // 4. Enable submit button
                if (submitBtn) {
                    submitBtn.disabled = false;
                }

                // 5. Update status
                updateStatus('Device information captured successfully!', 'success');

                console.log('✅ All device info set in form');

            } catch (error) {
                console.error('Error in device capture:', error);
                updateStatus('Error capturing device info', 'danger');
                return;
            }

            // 6. Form submission handler
            form.addEventListener('submit', async function(e) {
                e.preventDefault();
                console.log('Form submitted');

                if (submitBtn) {
                    submitBtn.disabled = true;
                    const submitText = document.getElementById('submitText');
                    const submitSpinner = document.getElementById('submitSpinner');

                    if (submitText) submitText.textContent = 'Processing...';
                    if (submitSpinner) submitSpinner.style.display = 'inline-block';
                }

                updateStatus('Submitting registration...', 'warning');

                try {
                    // Get fresh device info
                    const freshDeviceInfo = await deviceCapture.getAllDeviceInfo();
                    setDeviceInfo(freshDeviceInfo);

                    // Create FormData
                    const formData = new FormData(this);

                    // Log what we're sending
                    console.log('📤 Form Data Being Sent:');
                    let hasAllRequiredFields = true;
                    const requiredFields = [
                        'device_data[ip_address]',
                        'device_data[browser_fingerprint]',
                        'device_data[operating_system]',
                        'device_data[browser]',
                        'device_data[device_name]',
                        'device_data[device_type]'
                    ];

                    for (let [key, value] of formData.entries()) {
                        console.log(`${key}: ${value ? '✅' : '❌'} "${value}"`);
                        if (requiredFields.includes(key) && !value) {
                            hasAllRequiredFields = false;
                            console.error(`Missing required field: ${key}`);
                        }
                    }

                    if (!hasAllRequiredFields) {
                        throw new Error('Missing required device fields');
                    }

                    // Send request
                    const response = await fetch('/api/users', {
                        method: 'POST',
                        body: formData,
                        headers: {
                            'Accept': 'application/json',
                            'X-Requested-With': 'XMLHttpRequest'
                        }
                    });

                    const result = await response.json();
                    console.log('📥 Server Response:', result);

                    // Reset button
                    if (submitBtn) {
                        submitBtn.disabled = false;
                        const submitText = document.getElementById('submitText');
                        const submitSpinner = document.getElementById('submitSpinner');

                        if (submitText) submitText.textContent = 'Register User';
                        if (submitSpinner) submitSpinner.style.display = 'none';
                    }

                    if (result.success) {
                        updateStatus('Registration completed!', 'success');
                        showResponse(true, result.message, result.data);

                        // Reset form but keep device info
                        form.reset();
                        // Re-set device info
                        setDeviceInfo(freshDeviceInfo);
                    } else {
                        updateStatus('Registration failed', 'danger');
                        showResponse(false, result.message, result);
                    }

                } catch (error) {
                    console.error('❌ Submission Error:', error);

                    // Reset button
                    if (submitBtn) {
                        submitBtn.disabled = false;
                        const submitText = document.getElementById('submitText');
                        const submitSpinner = document.getElementById('submitSpinner');

                        if (submitText) submitText.textContent = 'Register User';
                        if (submitSpinner) submitSpinner.style.display = 'none';
                    }

                    updateStatus('Network error occurred', 'danger');
                    showResponse(false, 'Network error: ' + error.message);
                }
            });
        });
    </script>
</body>
</html>
